package cz.drg.clasificator.writers;

import cz.drg.clasificator.args.argevaluation.ConfigurableIOClass;
import cz.drg.clasificator.exception.ShutdownException;
import cz.drg.clasificator.setting.program.IOclass;
import cz.drg.clasificator.util.Constants;
import cz.drg.clasificator.util.HeaderList;
import cz.drg.clasificator.util.InsertQueryBatch;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;
import java.util.Scanner;
import static cz.drg.clasificator.util.OutputHelper.*;


public class DBWriter implements OutputWriter, ConfigurableIOClass{

    //determines if the application is executed as a service
    private boolean isService = false;
    
    private IOclass ioClass;
    private String username;
    private String password;
    private Connection connection;
    
    @Override
    public void writeOutput(HeaderList originalInput, List<List<String>> lines) {
        
        if(!isConnected()){
            connection = createNewConnection();
        }
        
        InsertQueryBatch insertQueryBatch = new InsertQueryBatch(ioClass);
        try {
            //for Oracle set config for current connection
            prepareForSpecificDbVendors(connection);
            
            insertQueryBatch.createQueries(originalInput, lines);
            Statement filledStatement = insertQueryBatch.fillStatement(connection.createStatement());
            filledStatement.executeBatch();
            filledStatement.close();
            
            disconnect();
        } catch (SQLException ex) {
            
            errLog("Insert batch was:");
            errLog(insertQueryBatch.toString());
            errLog(ex.getMessage());
            disconnect();
            
            throw new ShutdownException(Constants.ERR_SQL_WRITE);
        } 
    }
    
    @Deprecated
    @Override
    public void writeOutput(List<String> lines) {
        throw new UnsupportedOperationException("Not supported.");
    }
    
    private Connection createNewConnection(){
        
        String databaseUrl = getTargetDatabaseUrl();
        errLog("Creating new connection to database "+databaseUrl+" for writing.");
        
        if(username != null && password != null){
            return connect(databaseUrl);
        }
        
        if(hasDBCredentialsInConfig()){
            username = ioClass.getDatabaseUser();
            password = ioClass.getDatabasePassword();
        }
        //if there are no db credentials in configuration and it was not already requested manualy
        else {
            Scanner scan = new Scanner(System.in);
            dualLog("Enter username:");
            username = scan.nextLine();
            dualLog("Enter password:");
            password = scan.nextLine();
        }
        
        return connect(databaseUrl);
    }
    
    private void prepareForSpecificDbVendors(Connection conn) throws SQLException{
        //oracle has some default SESSION settings that derives from NLS_LANG = AMERICA
        //but on czech systems the formats are different from AMERICA formats
        
        if(ioClass.getDatabaseConnection().contains("jdbc:oracle:thin:@")){
            //set SESSION specific number formatting before inserting the values
            PreparedStatement prepareStatement = conn.prepareStatement("ALTER SESSION SET NLS_NUMERIC_CHARACTERS = '.,'");
            prepareStatement.execute();
            prepareStatement.close();
        }
    }
    
    @Override
    public void setIoClass(IOclass config) {
        if(config == null){
            dualLog("No database configuration set for this class. Add <databaseConnection> setting into the related <IOclass> setting first!");
            System.exit(-1);
        }
        this.ioClass = config;
    }

    @Override
    public void clear() {
        
        if(!isConnected()){
            connection = createNewConnection();
        }
        
        if(!isService){
        
            try {
                PreparedStatement prepareStatement = connection.prepareStatement("TRUNCATE TABLE "+getTargetTable());
                prepareStatement.executeUpdate();
                prepareStatement.close();
            } catch (SQLException ex) {
                
                dualLog(ex.getMessage());
                        
                throw new ShutdownException(Constants.ERR_SQL_CLEAR);
            }
            finally{
                disconnect();
            }
            
        }
        
    }

    @Override
    public void close() {
        disconnect();
    }

    private Connection connect(String databaseUrl){
        Connection connection = null;
        
        try {
            
            connection = DriverManager.getConnection(databaseUrl, username, password);
            connection.setAutoCommit(true);
            
        } catch (SQLException ex) {
            
            dualLog(ex.getMessage());
            throw new ShutdownException(String.format(Constants.ERR_SQL_CONNECTION, databaseUrl));
            
        } 
        
        return connection;
    }
    
    private void disconnect(){
        if(connection == null) return;
        
        try {
            connection.close();
            connection = null;
        } catch (SQLException ex) {
            
            dualLog(ex.getMessage());
            throw new ShutdownException(Constants.ERR_SQL_DISCONNECT);
            
        }
    }
    
    private String getTargetDatabaseUrl(){
        return ioClass.getDatabaseConnection();
    }
    
    private String getTargetTable(){
        return ioClass.getDatabaseTargetTable();
    }
    
    private boolean isConnected(){
        return connection != null;
    }
    
    private boolean hasDBCredentialsInConfig(){
        return ioClass.getDatabaseUser() != null && ioClass.getDatabasePassword() != null;
    }

    /**
     * @return true if the application was executed with a parameter "-service". If it was, contents of target database are not truncated before insertion of evaluated rows.
     */
    public boolean isService() {
        return isService;
    }
    
    public void setIsService(boolean isService){
        this.isService = isService;
    }

}
